<?php
session_start();
require "config.php";
require __DIR__ . '/vendor/autoload.php';

use MercadoPago\MercadoPagoConfig;
use MercadoPago\Client\Payment\PaymentClient;

// Configurar token do Mercado Pago (substitua pela sua chave de acesso)
MercadoPagoConfig::setAccessToken($mercadoPagoAccessToken);

// Pega UUID da URL
$uuid = isset($_GET['uuid']) ? $_GET['uuid'] : null;

if (!$uuid) {
    http_response_code(403);
    include __DIR__ . '/errordocument/403.php';
    exit;
}

// Busca no banco pelo UUID com JOIN para pegar nome e email
$stmt = $pdo->prepare("
    SELECT p.*, pn.fullname, pn.email 
    FROM payments p 
    LEFT JOIN payments_names pn ON p.account_id = pn.account_id 
    WHERE p.uuid = ?
");
$stmt->execute([$uuid]);
$payment = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$payment) {
    http_response_code(403);
    include __DIR__ . '/errordocument/403.php';
    exit;
}

// Variáveis do banco
$statusNumeric = $payment['status'];
$value = (float) $payment['value'] / 100;
$pixCode = $payment['pix_code'];
$qrBase64 = $payment['qr_code'];
$expire = $payment['expire'];
$paid = $payment['paid'];
$mpPaymentId = $payment['mp_payment_id'] ?? null;
$accountId = $payment['account_id'];

// Dados do usuário para o pagamento
$userFullname = $payment['fullname'] ?? 'Usuário Anônimo';
$userEmail = $payment['email'] ?? 'anonimo@example.com';

$client = new PaymentClient();

function criarPagamentoPixMercadoPago(PaymentClient $client, $value, $fullname, $email) {
    // Separar nome e sobrenome
    $nameParts = explode(' ', trim($fullname), 2);
    $firstName = $nameParts[0] ?? 'Usuario';
    $lastName = isset($nameParts[1]) ? $nameParts[1] : 'Anonimo';
    
    // Validar email
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $email = 'anonimo@example.com';
    }
    
    $paymentRequest = [
        "transaction_amount" => (float) $value,
        "description" => "Donate via Pix",
        "payment_method_id" => "pix",
        "payer" => [
            "email" => $email,
            "first_name" => $firstName,
            "last_name" => $lastName,
            "identification" => [
                "type" => "CPF",
                "number" => "00000000000" // Você pode coletar o CPF se necessário
            ]
        ]
    ];

    try {
        $payment = $client->create($paymentRequest);
        return $payment;
    } catch (Exception $e) {
        error_log("Erro ao criar pagamento Mercado Pago: " . $e->getMessage());
        return null;
    }
}

function obterPixMercadoPago(PaymentClient $client, $mpPaymentId) {
    try {
        $payment = $client->get($mpPaymentId);

        $pixCode = $payment->point_of_interaction->transaction_data->qr_code ?? null;
        $qrBase64 = $payment->point_of_interaction->transaction_data->qr_code_base64 ?? null;

        return [$pixCode, $qrBase64];
    } catch (Exception $e) {
        error_log("Erro ao buscar pagamento Mercado Pago: " . $e->getMessage());
        return [null, null];
    }
}

// Se não tem mp_payment_id, cria o pagamento no Mercado Pago
if (!$mpPaymentId) {
    $paymentMP = criarPagamentoPixMercadoPago($client, $value, $userFullname, $userEmail);

    if ($paymentMP && isset($paymentMP->id)) {
        $mpPaymentId = $paymentMP->id;

        // Atualiza o banco com o ID do pagamento do Mercado Pago
        $stmt = $pdo->prepare("UPDATE payments SET mp_payment_id = ?, at_payment_update = NOW() WHERE uuid = ?");
        $stmt->execute([$mpPaymentId, $uuid]);
    }
}

// Se está pendente e tem mp_payment_id mas não tem PIX code/QR code
if ($statusNumeric === 0 && $mpPaymentId && (empty($pixCode) || empty($qrBase64))) {
    list($newPixCode, $newQrBase64) = obterPixMercadoPago($client, $mpPaymentId);

    if ($newPixCode && $newQrBase64) {
        $stmt = $pdo->prepare("UPDATE payments SET pix_code = ?, qr_code = ?, at_payment_update = NOW() WHERE uuid = ?");
        $stmt->execute([$newPixCode, $newQrBase64, $uuid]);

        $pixCode = $newPixCode;
        $qrBase64 = $newQrBase64;
    }
}

// Ajuste timezone conforme seu servidor
date_default_timezone_set('America/Sao_Paulo');
$now = new DateTime();
$expireDate = DateTime::createFromFormat('Y-m-d H:i:s', $expire);
if (!$expireDate) {
    $isExpired = false;
} else {
    $isExpired = $now > $expireDate;
}

// Ajusta status para expirado se aplicável
$status = '';
switch ($statusNumeric) {
    case 0: $status = 'pending'; break;
    case 1: $status = 'completed'; break;
    case 2: $status = 'canceled'; break;
    default: $status = 'unknown'; break;
}
if ($isExpired && $status === 'pending') {
    $status = 'expired';
}

// Controle para exibir popup só uma vez por sessão
$showPopup = false;
if ($statusNumeric == 1 && empty($_SESSION['popup_exibido_' . $uuid])) {
    $showPopup = true;
    $_SESSION['popup_exibido_' . $uuid] = true;
}
?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8" />
    <title>Doação <?php echo htmlspecialchars($serverName); ?></title>
    <link rel="stylesheet" href="style.css" />
</head>
<body>
    <h1><?php echo htmlspecialchars($serverName); ?></h1>

    <div class="pix-box">
        <p><strong>Valor:</strong> R$ <?php echo number_format($value, 2, ',', '.'); ?></p>
        <p><strong>Status:</strong> 
            <?php if ($status === 'pending'): ?>
                <span class="pending">Pendente</span>
            <?php elseif ($status === 'completed'): ?>
                <span class="completed">Confirmado</span>
            <?php elseif ($status === 'canceled'): ?>
                <span class="canceled">Cancelado</span>
            <?php elseif ($status === 'expired'): ?>
                <span class="expired">Expirado</span>
            <?php else: ?>
                <span class="unknown">Desconhecido</span>
            <?php endif; ?>
        </p>

        <?php if (!empty($userFullname) && $userFullname !== 'Usuário Anônimo'): ?>
            <p><strong>Usuário:</strong> <?php echo htmlspecialchars($userFullname); ?></p>
        <?php endif; ?>

        <?php if (!$isExpired): ?>
            <p id="resgate"><strong>Resgate:</strong>
                <?php 
                if ($paid == 0) { echo " Pendente"; } 
                elseif ($paid == 1) { echo " Cancelado"; } 
                elseif ($paid == 2) { echo " Aprovado"; } 
                elseif ($paid == 3) { echo " RoDEx"; } 
                else { echo " Desconhecido";}
                ?>
            </p>
        <?php endif; ?>

        <?php if ($status === 'pending' && $pixCode && !$isExpired): ?>
            <p><strong>Expira em:</strong> <span id="countdown"></span></p>

            <div class="qr">
                <img src="data:image/png;base64,<?php echo $qrBase64; ?>" alt="QR Code Pix" />
            </div>

            <p id="copy-label"><strong>Copia e Cola:</strong></p>
            <input type="text" readonly value="<?php echo htmlspecialchars($pixCode); ?>" onclick="this.select();document.execCommand('copy');alert('Código copiado!')" />
        <?php elseif ($status === 'expired'): ?>
            <p class="canceled" id="expired-message">Este pagamento expirou.</p>
        <?php endif; ?>
    </div>

    <!-- Popup de agradecimento -->
    <div class="popup-overlay" id="popupAgradecimento">
        <div class="popup">
            <h2>Obrigado!</h2>
            <p><?php echo htmlspecialchars($mensagemAgradecimento); ?></p>
            <button id="btnFecharPopup">Fechar</button>
        </div>
    </div>

    <footer>
        <p>As doações realizadas têm caráter voluntário e não são reembolsáveis. Os valores arrecadados serão destinados exclusivamente à manutenção e desenvolvimento do servidor, sem gerar qualquer obrigação contratual ou vínculo comercial.</p>
        <p>&copy; 2025 <?php echo htmlspecialchars($serverName); ?>. Todos os direitos reservados.</p>
    </footer>

    <script>
        // Contador de expiração
        const expireTime = new Date("<?php echo $expire; ?>").getTime();
        const countdownEl = document.getElementById("countdown");

        function updateCountdown() {
            const now = new Date().getTime();
            const diff = expireTime - now;

            if (diff <= 0) {
                if(countdownEl) countdownEl.textContent = "";

                const pixBox = document.querySelector('.pix-box');
                if(!pixBox) return;

                const qrDiv = pixBox.querySelector('.qr');
                const copyInput = pixBox.querySelector('input[readonly]');
                const copyLabel = document.getElementById('copy-label');
                const resgateP = document.getElementById('resgate');
                const expiredMessage = document.getElementById('expired-message');

                if (qrDiv) qrDiv.classList.add('hidden');
                if (copyInput) copyInput.classList.add('hidden');
                if (copyLabel) copyLabel.classList.add('hidden');
                if (resgateP) resgateP.classList.add('hidden');

                if (!expiredMessage) {
                    const expiredMsg = document.createElement('p');
                    expiredMsg.id = 'expired-message';
                    expiredMsg.className = 'canceled';
                    expiredMsg.textContent = 'Este pagamento expirou.';
                    pixBox.appendChild(expiredMsg);
                }
                return;
            }

            if(countdownEl) {
                const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                const seconds = Math.floor((diff % (1000 * 60)) / 1000);
                countdownEl.textContent = `${minutes}m ${seconds}s`;
            }
            setTimeout(updateCountdown, 1000);
        }

        if(countdownEl) updateCountdown();

        // Popup automático de agradecimento
        const popup = document.getElementById('popupAgradecimento');
        const btnFechar = document.getElementById('btnFecharPopup');

        btnFechar.addEventListener('click', () => {
            popup.style.display = 'none';
            location.reload(); // Mantém o reload após fechar, se desejado
        });

        // Verificação periódica do status via AJAX
        let currentStatus = '<?php echo $status; ?>';
        let popupShown = <?php echo $showPopup ? 'true' : 'false'; ?>;

        function checkPaymentStatus() {
            if (currentStatus === 'completed' || currentStatus === 'expired' || currentStatus === 'canceled' || popupShown) {
                return; // Para de checar se já finalizado ou popup já exibido
            }

            fetch('check_status.php?uuid=<?php echo $uuid; ?>')
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'completed' && !popupShown) {
                        currentStatus = 'completed';
                        popup.style.display = 'flex';
                        popupShown = true;
                        // Opcional: Atualizar o status na página sem reload
                        document.querySelector('.pix-box p:nth-child(2) span').textContent = 'Confirmado';
                        document.querySelector('.pix-box p:nth-child(2) span').className = 'completed';
                        // Esconder elementos de pagamento pendente
                        const qrDiv = document.querySelector('.qr');
                        const copyInput = document.querySelector('input[readonly]');
                        const copyLabel = document.getElementById('copy-label');
                        const countdownP = document.querySelector('p:has(#countdown)');
                        const resgateP = document.getElementById('resgate');
                        if (qrDiv) qrDiv.classList.add('hidden');
                        if (copyInput) copyInput.classList.add('hidden');
                        if (copyLabel) copyLabel.classList.add('hidden');
                        if (countdownP) countdownP.classList.add('hidden');
                        if (resgateP) resgateP.innerHTML = '<strong>Resgate:</strong> Aprovado'; // Atualiza resgate se necessário
                    } else if (data.status === 'expired') {
                        currentStatus = 'expired';
                        // Atualizar para expirado
                        document.querySelector('.pix-box p:nth-child(2) span').textContent = 'Expirado';
                        document.querySelector('.pix-box p:nth-child(2) span').className = 'expired';
                        // Esconder elementos
                        const qrDiv = document.querySelector('.qr');
                        const copyInput = document.querySelector('input[readonly]');
                        const copyLabel = document.getElementById('copy-label');
                        const countdownP = document.querySelector('p:has(#countdown)');
                        const resgateP = document.getElementById('resgate');
                        if (qrDiv) qrDiv.classList.add('hidden');
                        if (copyInput) copyInput.classList.add('hidden');
                        if (copyLabel) copyLabel.classList.add('hidden');
                        if (countdownP) countdownP.classList.add('hidden');
                        if (resgateP) resgateP.classList.add('hidden');
                        const expiredMsg = document.createElement('p');
                        expiredMsg.className = 'expired';
                        expiredMsg.textContent = 'Este pagamento expirou.';
                        document.querySelector('.pix-box').appendChild(expiredMsg);
                    }
                })
                .catch(error => console.error('Erro ao checar status:', error));
        }

        // Inicia a verificação a cada 5 segundos (pode ajustar para 3 ou outro valor)
        if (currentStatus === 'pending') {
            setInterval(checkPaymentStatus, 5000);
        }

        // Exibe popup inicial se já completado no load
        <?php if ($showPopup): ?>
        popup.style.display = 'flex';
        <?php endif; ?>
    </script>
</body>
</html>